#!/bin/bash
# revision: 2024/2/14: initial version

set -o nounset

SCRIPT_NAME=$(basename "$0")
DUT_IP=""
DUT_USER=""
DUT_PASS=""
IMAGE=""
# SPI0(num = 1) and SPI1(num = 2)
# default select SPI0
SPI_NUM=1
OPTIONS=""

#CMD_YAFU=./Yafuflash
CMD_IPMI=ipmitool
CMD_JQ=jq
CMD_CURL=curl

function LOG()
{
    msg=$1
    echo "[`date +%Y%m%d-%H%M`][INFO ] $msg" 
}

function ERROR()
{
    msg=$1
    echo "[`date +%Y%m%d-%H%M`][ERROR] $msg" 
    exit;
}

function PROMPT()
{
    msg=$1
    echo -n "[`date +%Y%m%d-%H%M`][WARN ] $msg: " 
}

function usage()
{
	echo "Usage"
	#echo "./$SCRIPT_NAME <info|update|help> <-I [ip]|-U [user name]|-P [password]|-F [file]|-s [0|1|2]|-o [...]>"
	echo "./$SCRIPT_NAME <update|help> <-I [ip]|-U [user name]|-P [password]|-F [file]>"
	echo "  help    : This help."
	#echo "  info    : Query DUT image information, total we have image1 and image2."
	echo "  update  : firmware upgrade."
	echo "  -I      : specify DUT ip address."
	echo "  -U      : specify DUT login user name."
	echo "  -P      : specify DUT login password."
	echo "  -F      : specify firmware file to upgrade."
	#echo "  -s      : specify image number to upgrade, 0=inactive image, 1=image1, 2=image2"
	#echo "            0=inactive image."
	#echo "            1=image1."
	#echo "            2=image2."
	#echo "            image1 is default selected."
	#echo "  -o      : additional options,"
	#echo "            for example: \"-fb\" force bootloader upgrade during full upgrade."
}

function p_parse_args () {
    while [[ "$#" -gt 0 ]];
    do
        case "$1"
        in
            -I)
                DUT_IP="$2"
                LOG "DUT_IP = ${DUT_IP}"
                shift;;
            -P)
                DUT_PASS="$2"
                #LOG "DUT_PASS = ${DUT_PASS}"
                shift;;                                                                                  
            -U)
                DUT_USER="$2"
                LOG "DUT_USER = ${DUT_USER}"
                shift;;
            -F)
                IMAGE="$2"
                LOG "IMAGE = ${IMAGE}"
                shift;;
            -s)
                SPI_NUM="$2"
                LOG "SPI_NUM = ${SPI_NUM}"
                shift;;
            -o)
                OPTIONS="$2"
                LOG "OPTIONS = ${OPTIONS}"
                shift;;
 
            *)
                ERROR "Invalid option: \"$1\" !!";
                exit 0
        esac
        shift
    done
}

function connection_check()
{
	is_connection_ok=1
	ret=$(${CMD_IPMI} -C 17 -I lanplus -H $DUT_IP -U $DUT_USER -P $DUT_PASS mc info 2>&1 | grep "Error")
	if [ "x$ret" != "x" ];then
		ERROR "$ret"
	fi
}

function valid_ip()
{
    local  ip=$1
    local  stat=1

    if [[ $ip =~ ^[0-9]{1,3}\.[0-9]{1,3}\.[0-9]{1,3}\.[0-9]{1,3}$ ]]; then
        OIFS=$IFS
        IFS='.'
        ip=($ip)
        IFS=$OIFS
        [[ ${ip[0]} -le 255 && ${ip[1]} -le 255 && ${ip[2]} -le 255 && ${ip[3]} -le 255 ]]
        stat=$?
    fi
    echo $stat
}

function p_check ()
{
	if [ "x$DUT_IP" == "x" ];then
		read -p "Please enter DUT ip address: " ip
		ret=$(valid_ip $ip)
		if [ "x$ret" != "x0" ];then
			ERROR "Invalid DUT IP $ip"
		else
			DUT_IP=$ip
        	LOG "DUT_IP = ${DUT_IP}"
		fi
	fi
	if [ "x$DUT_USER" == "x" ];then
		read -p "Please enter DUT user name: " user
		if [ "x$user" == "x" ];then
			ERROR "Invalid DUT user name $user"
		else
			DUT_USER=$user
        	LOG "DUT_USER = ${DUT_USER}"
		fi
	fi
	if [ "x$DUT_PASS" == "x" ];then
		read -s -p "Please enter DUT password: " pass
		echo ""
		if [ "x$pass" == "x" ];then
			ERROR "Invalid DUT password $pass"
		else
			DUT_PASS=$pass
		fi
	fi
	if [ "x$SPI_NUM" != "x0" ] && [ "x$SPI_NUM" != "x1" ] && [ "x$SPI_NUM" != "x2" ] ;then
		ERROR "Invalid SPI_NUM $SPI_NUM"
	fi

}


function command_check()
{
	if ! [ -x "$(command -v $CMD_IPMI)" ]; then
		ERROR "$CMD_IPMI is not installed"
	fi
	if ! [ -x "$(command -v $CMD_JQ)" ]; then
		ERROR "$CMD_JQ is not installed"
	fi
	if ! [ -x "$(command -v $CMD_CURL)" ]; then
		ERROR "$CMD_CURL is not installed"
	fi
	#if [ ! -f $CMD_YAFU ]; then
	#	ERROR "command $CMD_YAFU is not exist, should be the same path as this script."
	#fi

}

function bmc_info()
{
	$CMD_YAFU -nw -ip $DUT_IP -u $DUT_USER -p $DUT_PASS -msi
}

function bmc_update()
{
    if [ ! -f $IMAGE ];then
        ERROR "File $IMAGE not found."
    fi
    LOG "Start BMC Update"
    #LOG " - SPI Num: $SPI_NUM"                                                     
    LOG " - DUT IP : $DUT_IP"
    LOG " - FW File: $IMAGE"
    LOG "====================="
    #echo "token: curl -k -H "Content-Type: application/json" -X POST https://${DUT_IP}/login -d '{"username" :  "${DUT_USER}", "password" :  "${DUT_PASS}"}' | grep token | awk '{print $2;}' | tr -d '"'"
    token=$(curl -k -H "Content-Type: application/json" -X POST https://${DUT_IP}/login -d "{\"username\" :  \"$DUT_USER\", \"password\" :  \"$DUT_PASS\"}" | grep token | awk '{print $2;}' | tr -d '"')
    echo "token: $token"
    echo "uri: curl -k -H \"X-Auth-Token: $token\" https://${DUT_IP}/redfish/v1/UpdateService | jq -r ' .HttpPushUri'"
    uri=$(curl -k -H "X-Auth-Token: $token" https://${DUT_IP}/redfish/v1/UpdateService | jq -r ' .HttpPushUri')
    #echo "curl -k -H \"X-Auth-Token: $token\" -H \"Content-Type: application/octet-stream\" -X POST -T ${IMAGE} https://${DUT_IP}${uri}"
    curl -k -H "X-Auth-Token: $token" -H "Content-Type: application/octet-stream" -X POST -T ${IMAGE} https://${DUT_IP}${uri}
    #$CMD_YAFU -nw -ip $DUT_IP -u $DUT_USER -p $DUT_PASS $OPTIONS -mse $SPI_NUM $IMAGE
}




if [ $# -lt 1 ];then
	usage
	exit -1;
fi

command_check

while [ $# -gt 0 ];do
	a=$1
	shift
	case "$a" in
	    help)
			usage
	        ;;
		#info)
		#	p_parse_args $@
		#	p_check
		#	bmc_info
		#	exit;;
		update)
			p_parse_args $@
			p_check
			connection_check
			bmc_update
			exit;;
	    *)
			ERROR "Please enter option in [help|info|update]"
	        usage
	        ;;
	esac
done
